const axios = require('axios')
const Tenant = require('./models/Tenant')
const passport = require('passport');
const { Strategy } = require('passport-openidconnect');
const logger = require('./logger')
const atob = require('atob');
const {IDPType} = require('./models/IDPType')

const ResolutionStrategy = {
    NONE: 'none',
    SUBDOMAIN: 'subdomain',
    QUERY: 'query'
}

const defaultTenantSub = "default"
const baseHost = new URL(process.env.BASE_URI).hostname
const resolutionStrategy = process.env.RESOLUTION_STRATEGY || ResolutionStrategy.SUBDOMAIN
const demoApiEndpoint = process.env.DEMO_API_ENDPOINT || "https://api.demo.okta.com"
const demoApiAudience = process.env.DEMO_API_AUDIENCE || "https://api.demo.okta.com"
const demoApiTokenEndpoint = process.env.DEMO_API_TOKEN_ENDPOINT || "https://auth.demo.okta.com/oauth/token"

class TenantResolver {
    constructor() {
        this.tenants = new Map([]);
        if (process.env.DEFAULT_ISSUER && process.env.DEFAULT_CLIENT_ID && process.env.DEFAULT_CLIENT_SECRET) {
            logger.info("Default config found this will be used for " + process.env.BASE_URI)
            this.tenants.set(defaultTenantSub, new Tenant(defaultTenantSub))
            this.configureTenantStrategy(this.tenants.get(defaultTenantSub), defaultTenantSub)
        }
        if (!process.env.DEMO_API_APP_ID || !process.env.DEMO_API_CLIENT_ID || !process.env.DEMO_API_CLIENT_SECRET) {
            logger.warn("Missing environment variables for Demo API, configuration cannot be dynamically retrieved.")
        }
    }

    getTenant(request) {
        let tenant
        switch (resolutionStrategy) {
            case ResolutionStrategy.SUBDOMAIN:
                tenant = request.headers.host.substr(0, request.headers.host.indexOf("." + baseHost))
                if (tenant === '') {
                    return defaultTenantSub
                }
                return tenant
            case ResolutionStrategy.QUERY:
                //prioritize the query param to reset
                if(request.query.demoName){
                    tenant = request.query.demoName
                }
                //use the cookie if the query param has been cleared
                else if (request.cookies.demo){
                    tenant = request.cookies.demo
                }
                if (tenant === undefined) {
                    return defaultTenantSub
                }
                return tenant
            default:
                return defaultTenantSub
        }

    }

    getSettings(tenant) {
        return this.tenants.get(tenant)
    }

    resolveTenant() {
        return async (req, res, next) => {
            let sub = this.getTenant(req)
            var url = new URL(process.env.BASE_URI)
            if(resolutionStrategy === ResolutionStrategy.QUERY){
                res.cookie('demo',sub,{domain:'.'+url.host})
            }

            logger.verbose("Request for subdomain '" + sub + "' received.")
            var tenant = this.tenants.get(sub)
            //skip resolution if using default and it was not setup at construct
            if(tenant == null && sub === defaultTenantSub){
                next()
                return
            }

            if (tenant == null || tenant.isExpired()) {
                try {
                    //perform demo API bootstrap
                    logger.info("Consulting Demo API for tenant info of " + sub)
                    var bootstrapResponse = await axios.get(demoApiEndpoint + "/bootstrap/" + process.env.DEMO_API_APP_ID + "/" + sub, {
                        headers: {
                            Authorization: 'Bearer ' + await this.getServiceToken()
                        }
                    })
                    const tenantBootstrap = bootstrapResponse.data

                    if (tenantBootstrap.settings.feature_custom_authorization_server_id) {
                        logger.info('Custom authorization server detected.');
            
                        tenantBootstrap.oidc_configuration.discoveryUrl = {
                          custom: tenantBootstrap.oidc_configuration.discoveryUrl.custom
                            ? `https://${
                                new URL(
                                  tenantBootstrap.oidc_configuration.discoveryUrl.custom
                                ).host
                              }/oauth2/${
                                tenantBootstrap.settings
                                  .feature_custom_authorization_server_id
                              }/.well-known/openid-configuration`
                            : undefined,
                          default: `https://${
                            new URL(tenantBootstrap.oidc_configuration.discoveryUrl.default)
                              .host
                          }/oauth2/${
                            tenantBootstrap.settings.feature_custom_authorization_server_id
                          }/.well-known/openid-configuration`,
                        };
                    }

                    //perform OIDC discovery
                    var discoveryDocument
                    if (tenantBootstrap.oidc_configuration.discoveryUrl) {
                        try {
                            logger.info("Retrieving OIDC configuration from well-known")
                            const discoveryUrl = tenantBootstrap.oidc_configuration.discoveryUrl.custom ? tenantBootstrap.oidc_configuration.discoveryUrl.custom : tenantBootstrap.oidc_configuration.discoveryUrl.default
                            const discoveryResponse = await axios.get(discoveryUrl);
                            discoveryDocument = discoveryResponse.data;
                        } catch (err) {
                            console.log("OIDC discovery failed.", { sub, error: err })
                        }
                    } else{
                        logger.warn("No discovery url provided on bootstrap. Will use fallback OIDC configuration.")
                    }
                    this.tenants.set(sub, new Tenant(sub, tenantBootstrap, discoveryDocument));
                    logger.info("Tenant " + sub + " stored");
                    tenant = this.tenants.get(sub)
                    this.configureTenantStrategy(tenant, sub)
                }
                catch (error) {
                    logger.warn("Failed to bootstrap demo", { sub: sub, error: error })
                    if (error.response && error.response.status == 404) {
                        req.session.errorMsg = "Unable to bootstrap demo " + sub
                    }
                    return res.redirect('/error');
                }
            }

            if (tenant == null) {
                return res.redirect('/error');
            }
            req.session.tenant = tenant
            req.session.settings = tenant.settings
            req.session.currentVersion = tenant.version
            next()
        }
    }

    removeTenant(sub) {
        try {
            this.tenants.delete(sub)
            passport.unuse(sub)
        }
        catch (err) {
            logger.error("Unable to remove tenant " + sub, { error: err })
        }
    }

    async getServiceToken() {
        if (!this.demoApiToken || this.isExpired(this.demoApiToken)) {
            try {
                var resp = await axios({
                    method: 'post',
                    url: demoApiTokenEndpoint,
                    headers: {
                        'Accept': 'application/json',
                        'Content-Type': 'application/json',
                    },
                    data: {
                        "grant_type": "client_credentials",
                        "client_id": process.env.DEMO_API_CLIENT_ID,
                        "client_secret": process.env.DEMO_API_CLIENT_SECRET,
                        "scope": "bootstrap",
                        "audience": demoApiAudience,
                    }
                })
                this.demoApiToken = resp.data.access_token;
            } catch (err) {
                logger.error("Unable to retrieve service token.")
                logger.error(err)
            }
        }
        return this.demoApiToken
    }

    isExpired(token) {
        try {
            if (token != null) {
                var base64Url = token.split('.')[1];
                var base64 = base64Url.replace('-', '+').replace('_', '/');

                var payload = JSON.stringify(JSON.parse(atob(base64)), undefined, '\t');
                if (!payload.exp || payload.exp <= Date.now()) {
                    return true
                }
                else {
                    return false
                }
            }
        } catch (err) {
            logger.error(err)
        }
        return true
    }

    configureTenantStrategy(tenant, sub) {
        var callback = new URL(process.env.BASE_URI)
        if(resolutionStrategy === ResolutionStrategy.SUBDOMAIN){
            callback.hostname = `${sub}.${callback.hostname}`
        }
        callback.pathname = '/callback'

        let strat = new Strategy({
            issuer: tenant.issuer,
            authorizationURL: tenant.authorizationURL,
            tokenURL: tenant.tokenURL,
            userInfoURL: tenant.userInfoURL,
            clientID: tenant.clientID,
            clientSecret: tenant.clientSecret,
            callbackURL: callback.toString(),
            scope: process.env.SCOPES
        }, (iss, profile, context, idToken, accessToken, refreshToken, params, verified) => {
            var user = {
                'profile': profile._json,
                'tokens': {
                    'id_token': idToken,
                    'access_token': accessToken,
                    'refresh_token': refreshToken
                }
            }
            return verified(null, user);
        });

        strat.authorizationParams = (options) => {
            let params = {}
            params.scope = process.env.SCOPES
            switch (options.tenant?.idpType) {
                case IDPType.CIC:
                    params.audience = "https://"+options.tenant.host + "/api/v2/"
                    break;
                default:
                    break;
            }
            return params
        }

        passport.use(sub, strat);
    }
}

module.exports = TenantResolver
