const logger = require('../logger.js');
const { IDPType } = require('./IDPType.js');
const semver = require('semver')

class Tenant {

    constructor(sub, bootstrap, discoveryDocument) {
        if (bootstrap) {
            try {
                this.expires = new Date(new Date().getTime() + process.env.CACHE_DURATION * 60000);
                logger.verbose("Expires: " + this.expires)

                this.state = bootstrap.state
                let tenantTypeDiscoverySignal = bootstrap.oidc_configuration.discoveryUrl?.default ? bootstrap.oidc_configuration.discoveryUrl.default : bootstrap.oidc_configuration.issuer
                this.idpType = tenantTypeDiscoverySignal.indexOf('auth0') > 0 ? IDPType.CIC : IDPType.Okta

                if (discoveryDocument) {
                    this.issuer = discoveryDocument.issuer
                    this.authorizationURL = discoveryDocument.authorization_endpoint
                    this.tokenURL = discoveryDocument.token_endpoint
                    this.userInfoURL = discoveryDocument.userinfo_endpoint
                    this.logoutUrl = discoveryDocument.end_session_endpoint
                } else {
                    this.issuer = bootstrap.oidc_configuration.issuer
                    this.authorizationURL = bootstrap.oidc_configuration.authorizeUrl
                    this.tokenURL = bootstrap.oidc_configuration.tokenUrl
                    this.userInfoURL = bootstrap.oidc_configuration.userInfoUrl
                    if (this.idpType === IDPType.CIC) {
                        // cic tenant
                        this.logoutUrl = `${bootstrap.oidc_configuration.issuer}oidc/logout`;
                    } else if (this.idpType === IDPType.Okta) {
                        // ocis tenant
                        this.logoutUrl = `${bootstrap.oidc_configuration.issuer}/v1/logout`;
                    }
                }

                //handle the case where a discovery doc exists but a logout URL is not published
                if(this.logoutUrl){
                    this.logoutOidcCompliant = true
                } else {
                    if (this.idpType === IDPType.CIC) {
                        // cic tenant
                        this.logoutUrl = `${bootstrap.oidc_configuration.issuer}v2/logout`;
                        this.logoutOidcCompliant = false
                    } else if (this.idpType === IDPType.Okta) {
                        // ocis tenant
                        this.logoutUrl = `${bootstrap.oidc_configuration.issuer}/oauth2/v1/logout`;
                        this.logoutOidcCompliant = true
                    }
                }

                this.clientID = bootstrap.oidc_configuration.client_id
                this.clientSecret = bootstrap.oidc_configuration.client_secret
                this.host = new URL(bootstrap.oidc_configuration.issuer).hostname

                logger.verbose("Host:" + this.host)
                logger.verbose("Issuer: " + this.issuer)
                logger.verbose("AuthzUrl: " + this.authorizationURL)
                logger.verbose("TokenUrl: " + this.tokenURL)
                logger.verbose("UserInfoUrl: " + this.userInfoURL)
                logger.verbose("ClientID: " + this.clientID)
                if (this.clientSecret != null) {
                    logger.verbose("ClientSecret: --present--")
                }
                else {
                    logger.warn("ClientSecret: --absent--")
                }

                this.settings = bootstrap.settings
                this.version = semver.coerce(bootstrap.version) || "1.0.0"
            }
            catch (error) {
                logger.error('Error encountered parsing tenant config.', { error: error, sub: sub, tenantProfileJson: bootstrap, discoveryDocument });
            }
        }
        else {
            try {
                this.expires = null
                this.state = 'active'
                this.issuer = process.env.DEFAULT_ISSUER
                this.authorizationURL = process.env.DEFAULT_ISSUER + '/v1/authorize'
                this.tokenURL = process.env.DEFAULT_ISSUER + '/v1/token',
                this.userInfoURL = process.env.DEFAULT_ISSUER + '/v1/userinfo',
                this.clientID = process.env.DEFAULT_CLIENT_ID,
                this.clientSecret = process.env.DEFAULT_CLIENT_SECRET,
                this.settings = {}
                this.version = "1.0.0"
            }
            catch (error) {
                logger.error('Failed to configure tenant from environment defaults.', { error: error });
            }
        }
    }

    isExpired() {
        logger.verbose("Checking if tenant data is expired.")
        if (this.expires === null) {
            logger.verbose("Tenant data set to never expire.")
            return false
        }
        logger.verbose("Expiry timestamp " + this.expires)
        return new Date() > this.expires
    }
}

module.exports = Tenant