require('dotenv').config();
const express = require('express');
const session = require('express-session');
const passport = require('passport');
const okta = require('@okta/okta-sdk-nodejs');
const TenantResolver = require('./tenantResolver');
const {IDPType} = require('./models/IDPType')
const remoteHandlebars = require('express-remote-handlebars');
const logger = require('./logger');
const semver = require('semver')
const cookieParser = require('cookie-parser');

const PORT = process.env.PORT || 3000;
app = express();
app.use(express.json());
app.use('/static', express.static('static'));
app.use('/views', express.static('views'));
app.use('/scripts', express.static(`${__dirname}/node_modules/clipboard/dist/`));

const viewDir = process.env.BASE_URI + '/views/layouts/main.handlebars'
app.engine('handlebars', remoteHandlebars({ layout: viewDir, cacheControl: 'max-age=600, stale-while-revalidate=86400' }));
app.set('view engine', 'handlebars');

app.use(session({
  cookie: { httpOnly: true },
  secret: require('crypto').randomBytes(64).toString('hex'),
  saveUninitialized: false,
  resave: true,
}));
app.use(cookieParser());

app.use(passport.initialize({ userProperty: 'userContext' }));
app.use(passport.session());
passport.serializeUser((user, next) => {
  next(null, user);
});

passport.deserializeUser((obj, next) => {
  next(null, obj);
});

function ensureAuthenticated() {
  return async (req, res, next) => {
    if (req.isAuthenticated() && req.userContext != null) {
      return next();
    }
    res.redirect('/login');
  };
}

const tr = new TenantResolver();
const router = express.Router();

router.get('/', tr.resolveTenant(), async (req, res, next) => {
  logger.verbose('/ requested');
  let settings;
  if (req.session.settings) {
    settings = req.session.settings;
  }
  res.render('index', {
    layout: req.session.settings?.templateURL ? req.session.settings?.templateURL : viewDir,
    demoName: tr.getTenant(req),
    tenantSettings: settings,
    currentVersion: req.session.currentVersion,
    latestVersion: req.session.currentVersion ? (semver.major(process.env.npm_package_version ? process.env.npm_package_version : '1.0.0') > semver.major(req.session.currentVersion) ? process.env.npm_package_version : null) : null });
});

router.get('/protected', tr.resolveTenant(), ensureAuthenticated(), async (req, res, next) => {
  logger.verbose('/ requested');
  let accessToken; let
    profile;
  if (req.userContext && req.userContext.tokens && req.userContext.tokens.access_token) {
    accessToken = parseJWT(req.userContext.tokens.access_token);
  }
  if (req.userContext && req.userContext.tokens && req.userContext.tokens.id_token) {
    profile = parseJWT(req.userContext.tokens.id_token);
  }
  res.render('protected', {
    layout: req.session.settings?.templateURL ? req.session.settings?.templateURL : viewDir,
    profile,
    accessToken,
  });
});

router.get('/download', (req, res, next) => {
  // this allows the direct download of the src as a zip removing the need to make the repo public
  // the file at this location should be updated before deploy but never checked into git
  const file = process.env.BASE_URI + '/static/demoapi-node-quickstart.zip';
  res.redirect(file);
});

router.get('/login', tr.resolveTenant(), (req, res, next) => {
  passport.authenticate(tr.getTenant(req), { tenant: req.session.tenant })(req, res, next);
});

router.get('/callback', (req, res, next) => { 
  if(req.query.error) {
    req.session.errorMsg = req.query.error_description;
    return res.redirect('/error');
  }
  passport.authenticate(
    tr.getTenant(req),
    { successRedirect: '/protected', failureRedirect: '/error' },
  )(req, res, next);
});
router.get('/logout', ensureAuthenticated(), (req, res) => {
  logger.verbose('/logout requested');
  const tenantSettings = tr.getSettings(tr.getTenant(req));

  let protocol = 'http';
  if (req.secure) {
    protocol = 'https';
  } else if (req.get('x-forwarded-proto')) {
    protocol = req.get('x-forwarded-proto').split(',')[0];
  }
  const returnUrl = encodeURIComponent(`${protocol}://${req.headers.host}/`);

  let logoutRedirectUrl
  logger.verbose(tenantSettings.idpType + tenantSettings.logoutOidcCompliant)
  if (tenantSettings.idpType === IDPType.CIC && !tenantSettings.logoutOidcCompliant) {
      //cic /v2/logout handling
      logoutRedirectUrl = `${tenantSettings.logoutUrl}?returnTo=${returnUrl}&client_id=${encodeURIComponent(tenantSettings.clientID)}`
  } else {
      logoutRedirectUrl = `${tenantSettings.logoutUrl}?post_logout_redirect_uri=${returnUrl}&id_token_hint=${encodeURIComponent(req.session.passport.user.tokens.id_token)}`
  }
  req.session.destroy();
  res.redirect(logoutRedirectUrl);
});

router.get('/error', async (req, res, next) => {
  let msg = 'An error occured, unable to process your request.';
  if (req.session.errorMsg) {
    msg = req.session.errorMsg;
    req.session.errorMsg = null;
  }
  res.render('error', {
    layout: req.session.settings?.templateURL ? req.session.settings?.templateURL : viewDir,
    msg,
  });
});

/* 
 * =======================================================================================================================
 * = BELOW ARE LIGHTWEIGHT EXAMPLE IMPLEMENTATIONS OF ALL WEBHOOKS AVAILABLE TO APPLICATIONS.                            =
 * = THESE ARE CALLED FROM THE DEMO API ON THE FOLLOWING EVENTS:                                                         =
 * =    * When an Application is initially attached to a Demonstration (Requested)                                       =
 * =    * After an Application Instance has it's OIDC Credentials created (Create)                                       =
 * =    * After an Application Instance settings are updated (Update)                                                    =
 * =    * Before an Application Instance is deleted from a Demonstration (Destroy)                                       =
 * =======================================================================================================================
 */

/*
 * This is the Demo Platform Request Webhook for Applications.
 * It is fired when an instance of this application is initially associated with a demo
 * in the Demo Platform but BEFORE it's OIDC credentials are issued to it by the IDP Tenant.
 *
 * This hook is intended to be used to check if the application can be deployed successful. For example
 * if the application usage is limited to 10 deployments this hooks allows the application to reject an
 * 11th instance. Similarly the application may be targetted for only a specific Identity Solution.
 * This webhook allows the application to reject other Identity Solutions.
 *
 * The request contains a JSON payload that takes this format:
 *
 * <code>{
 *      "demonstration": {
 *        "name": "url safe name of the demonstration",
 *        "owner": "email address of the user the demo belongs to",
 *      },
 *      "idp": {
 *         "name": "the name of the identity provider",
 *         "type": "the type of the identity provider. Either 'customer-identity' for CIC, 'okta-customer-identity' for OCIS, or 'workforce-identity' for WIC",
 *         "state": "the status of the identity provider within the demo platform. Either 'active', 'queued' or 'error'",
 *         "source": "how the identity provider was created in the demo platform. Either 'imported' or 'provisioned'",
 *       },
 *       "application": {
 *         "version": "the version of the application that is being updated (i.e. what version of the applciation the application instance was been deployed with)",
 *       },
 *     }
 * </code>
 */
router.post('/hooks/request', async (req, res, next) => {
  logger.info('Received request webhook from demo.okta. Running pre-creation checking steps for application.');
  res.sendStatus(202);
});

/*
 * This is the Demo Platform Create Webhook for Applications.
 * It is fired when an instance of this application associated with a demo
 * in the Demo Platform has successfully had it's OIDC credentials issued
 * to it by the IDP Tenant.
 *
 * This hook is intended to be used to begin deploying configuration or
 * changes to the Tenant Identity Provider, and to Configure the application
 * itself as demonstrated in this function's implementation.
 *
 * The request contains a JSON payload that takes this format:
 *
 * <code>{
 *      "demonstration": {
 *        "name": "url safe name of the demonstration",
 *        "owner": "email address of the user the demo belongs to",
 *      },
 *      "idp": {
 *         "name": "the name of the identity provider",
 *         "type": "the type of the identity provider. Either 'customer-identity' for CIC, 'okta-customer-identity' for OCIS, or 'workforce-identity' for WIC",
 *         "state": "the status of the identity provider within the demo platform. Either 'active', 'queued' or 'error'",
 *         "source": "how the identity provider was created in the demo platform. Either 'imported' or 'provisioned'",
 *         "management_credentials": {
 *            "tokenEndpoint": URI for obtaining a token,
 *            "clientId": "Demo Platform M2M Client ID to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *            "clientSecret": "Demo Platform M2M Client Secret to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *         },
 *       },
 *       "application": {
 *         "version": "the version of the application that is being updated (i.e. what version of the applciation the application instance was been deployed with)",
 *         "oidc_configuration": {
 *           "issuer": "URI of the issuer of the token (the Identity Provider in this case)",
 *           "authorizeUrl": "URI for initating a authentication and authorization flow agsint the ",
 *           "tokenUrl": "URI for obtaining a token",
 *           "userInfoUrl": "URI for the OIDC User Information Endpoint",
 *           "client_id": "The OAuth Client ID of the application within the Identity Provider",
 *           "client_secret": "The OAuth Client Secret of the application within the Identity Provider",
 *         },
 *         "settings": { a free form key value object containg all application settings and their values. Values will be either the default on the application or the application instance's specific overrides },
 *       },
 *     }
 * </code>
 */
router.post('/hooks/create', async (req, res, next) => {
  logger.info('Received creation webhook from demo.okta. Running creation steps for application.');

  switch (req.body.idp.type) {
    case 'customer-identity':
      res.sendStatus(200);
      logger.debug('Applying configuration for CIC Tenant.');
      try {
        // TODO: Implement your Application's CIC configuration here
        logger.info(('Configuration applied successfully.'));
      } catch (err) {
        // TODO: Implement your error handling logic here
        logger.error('Failed to apply configuration', { error: err });
      }

      break;
    default:
      res.sendStatus(200);
      logger.debug('Applying configuration for WIC Tenant.');
      var orgUrl = new URL(req.body.idp.management_credentials.tokenEndpoint);
      orgUrl.pathname = '';
      try {
        const client = new okta.Client({
          orgUrl: orgUrl.toString(),
          authorizationMode: 'PrivateKey',
          clientId: req.body.idp.management_credentials.clientId,
          scopes: ['okta.groups.manage', 'okta.groups.read', 'okta.apps.read', 'okta.apps.manage'],
          privateKey: req.body.idp.management_credentials.clientJWKS.keys[0],
          keyId: 'demoplatform',
        });

        const application = await client.applicationApi.getApplication({appId:req.body.application.oidc_configuration.client_id})

        const everyoneGroup = (await client.groupApi.listGroups({ q: 'everyone', limit: 1 })).next()
        await client.applicationApi.assignGroupToApplication(
          {
            appId: application.id,
            groupId: (await everyoneGroup).value.id,
          });
        
        application.visibility.hide.web = false
        application.settings.oauthClient.initiate_login_uri = application.settings.oauthClient.client_uri+'login'
        application.settings.oauthClient.idp_initiated_login = {"mode":"SPEC","default_scope":[]}
        await client.applicationApi.replaceApplication({appId:application.id,application:application});
        logger.info(('Configuration applied successfully'));
      } catch (err) {
        logger.error('Failed to apply configuration', { error: err });
      }
      break;
  }
});

/*
 * This is the Demo Platform Update Webhook for Applications.
 * It is fired when an instance of this application associated with a demo
 * in the Demo Platform has had it's settings changed.
 *
 * This hook is intended to be used to redeploy configuration or
 * changes to the Tenant Identity Provider, and/or to reconfigure the application
 * itself as demonstrated in this function's implementation.
 *
 * The request contains a JSON payload that takes this format:
 *
 * <code>{
 *      "demonstration": {
 *        "name": "url safe name of the demonstration",
 *        "owner": "email address of the user the demo belongs to",
 *      },
 *      "idp": {
 *         "name": "the name of the identity provider",
 *         "type": "the type of the identity provider. Either 'customer-identity' for CIC, 'okta-customer-identity' for OCIS, or 'workforce-identity' for WIC",
 *         "state": "the status of the identity provider within the demo platform. Either 'active', 'queued' or 'error'",
 *         "source": "how the identity provider was created in the demo platform. Either 'imported' or 'provisioned'",
 *         "management_credentials": {
 *            "tokenEndpoint": URI for obtaining a token,
 *            "clientId": "Demo Platform M2M Client ID to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *            "clientSecret": "Demo Platform M2M Client Secret to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *         },
 *       },
 *       "application": {
 *         "version": "the version of the application that is being updated (i.e. what version of the applciation the application instance was been deployed with)",
 *         "oidc_configuration": {
 *           "issuer": "URI of the issuer of the token (the Identity Provider in this case)",
 *           "authorizeUrl": "URI for initating a authentication and authorization flow agsint the ",
 *           "tokenUrl": "URI for obtaining a token",
 *           "userInfoUrl": "URI for the OIDC User Information Endpoint",
 *           "client_id": "The OAuth Client ID of the application within the Identity Provider",
 *           "client_secret": "The OAuth Client Secret of the application within the Identity Provider",
 *         },
 *         "settings": { a free form key value object containg all application settings and their values. Values will be either the default on the application or the application instance's specific overrides },
 *       },
 *     }
 * </code>
 */
router.post('/hooks/update', async (req, res, next) => {
  logger.info('Received modification webhook from demo.okta. Running update steps for application.');

  if (req.body && req.body.demonstration && req.body.demonstration.name) {
    // this removes the demo from the cache so that the latest settings are pulled on next request
    // alternatively the tenant could be updated from the application.settings object in this hook
    tr.removeTenant(req.body.demonstration.name);
  }

  res.sendStatus(200);
});

/*
 * This is the Demo Platform Destroy Webhook for Applications.
 * It is fired when an instance of this application associated with a demo
 * in the Demo Platform is being deleted/removed/disassociated from the demonstration.
 *
 * This hook is intended to be used to clean up configuration or
 * changes made during other hooks to the Tenant Identity Provider, and/or to clean up the application
 * itself as demonstrated in this function's implementation.
 *
 * The request contains a JSON payload that takes this format:
 *
 * <code>{
 *      "demonstration": {
 *        "name": "url safe name of the demonstration",
 *        "owner": "email address of the user the demo belongs to",
 *      },
 *      "idp": {
 *         "name": "the name of the identity provider",
 *         "type": "the type of the identity provider. Either 'customer-identity' for CIC, 'okta-customer-identity' for OCIS, or 'workforce-identity' for WIC",
 *         "state": "the status of the identity provider within the demo platform. Either 'active', 'queued' or 'error'",
 *         "source": "how the identity provider was created in the demo platform. Either 'imported' or 'provisioned'",
 *         "management_credentials": {
 *            "tokenEndpoint": URI for obtaining a token,
 *            "clientId": "Demo Platform M2M Client ID to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *            "clientSecret": "Demo Platform M2M Client Secret to be used for making changes to the tenant (i.e. using Management API or depoy CLI)",
 *         },
 *       },
 *       "application": {
 *         "version": "the version of the application that is being updated (i.e. what version of the applciation the application instance was been deployed with)",
 *         "oidc_configuration": {
 *           "issuer": "URI of the issuer of the token (the Identity Provider in this case)",
 *           "authorizeUrl": "URI for initating a authentication and authorization flow agsint the ",
 *           "tokenUrl": "URI for obtaining a token",
 *           "userInfoUrl": "URI for the OIDC User Information Endpoint",
 *           "client_id": "The OAuth Client ID of the application within the Identity Provider",
 *           "client_secret": "The OAuth Client Secret of the application within the Identity Provider",
 *         },
 *         "settings": { a free form key value object containg all application settings and their values. Values will be either the default on the application or the application instance's specific overrides },
 *       },
 *     }
 * </code>
 */
router.post('/hooks/destroy', async (req, res, next) => {
  logger.info('Received tear down webhook from demo.okta. Running cleanup steps for application.');

  if (req.body && req.body.demonstration && req.body.demonstration.name) {
    // this removes the demo from the cache so if it is re-added the new configuration is used
    tr.removeTenant(req.body.demonstration.name);
  }

  res.sendStatus(200);
});

app.use(router);

app.listen(PORT, () => logger.info('Application started',{version: semver.clean(process.env.npm_package_version ? process.env.npm_package_version : '1.0.0')}));

function parseJWT(token) {
  if (token != null) {
    const base64Url = token.split('.')[1];
    const base64 = base64Url.replace(/-/g, '+').replace(/_/g, '/');
    const padded = base64 + '='.repeat((4 - (base64.length % 4)) % 4);
    try {
      return JSON.parse(Buffer.from(padded, 'base64').toString('utf8'));
    } catch (err) {
      logger.warn(err)
      return {'Error':'Invalid token was parsed'};
    }
  } else {
    return {'Error':'Empty token was sent to be parsed'};
  }
}
